package handlers

import (
	"database/sql"
	"fmt"
	htmlTemplate "html/template"
	"net/http"
	"os/exec"
	"strings"
	textTemplate "text/template"

	"github.com/sourcegraph/security-coding-interview/internal/database"
	"github.com/sourcegraph/security-coding-interview/internal/models"
)

type Handler struct {
	router *database.Router
}

func NewHandler(router *database.Router) *Handler {
	return &Handler{router: router}
}

func (h *Handler) LoginPage(w http.ResponseWriter, req *http.Request) {
	tmpl := textTemplate.Must(textTemplate.ParseFiles("templates/login.html"))
	tmpl.Execute(w, nil)
}

func (h *Handler) Login(w http.ResponseWriter, req *http.Request) {
	username := req.FormValue("username")
	password := req.FormValue("password")

	var userID int
	var dbPassword, role string
	err := h.router.GetDB().QueryRow("SELECT id, password, role FROM users WHERE username = ?", username).
		Scan(&userID, &dbPassword, &role)

	if err != nil || dbPassword != password {
		http.Redirect(w, req, "/?error=invalid", http.StatusSeeOther)
		return
	}

	http.SetCookie(w, &http.Cookie{
		Name:  "user",
		Value: fmt.Sprintf("%d", userID),
		Path:  "/",
	})
	http.SetCookie(w, &http.Cookie{
		Name:  "session",
		Value: "authenticated",
		Path:  "/",
	})
	http.Redirect(w, req, "/dashboard", http.StatusSeeOther)
}

func (h *Handler) Dashboard(w http.ResponseWriter, req *http.Request) {
	if !h.router.IsAuthenticated(req) {
		http.Redirect(w, req, "/", http.StatusSeeOther)
		return
	}

	data := h.router.GetPageData(req)

	tmpl := htmlTemplate.Must(htmlTemplate.ParseFiles("templates/dashboard.html"))
	tmpl.Execute(w, data)
}

func (h *Handler) Users(w http.ResponseWriter, req *http.Request) {
	if !h.router.IsAuthenticated(req) {
		http.Redirect(w, req, "/", http.StatusSeeOther)
		return
	}

	pageData := h.router.GetPageData(req)
	search := req.URL.Query().Get("search")
	var query string
	var rows *sql.Rows
	var err error

	if search != "" {
		query = "SELECT id, username, role FROM users WHERE username LIKE '%" + search + "%'"
		rows, err = h.router.GetDB().Query(query)
	} else {
		rows, err = h.router.GetDB().Query("SELECT id, username, role FROM users")
	}

	if err != nil {
		http.Error(w, err.Error(), http.StatusInternalServerError)
		return
	}
	defer rows.Close()

	var users []models.User
	for rows.Next() {
		var u models.User
		rows.Scan(&u.ID, &u.Username, &u.Role)
		users = append(users, u)
	}

	tmpl := htmlTemplate.Must(htmlTemplate.ParseFiles("templates/users.html"))
	data := struct {
		models.PageData
		Users  []models.User
		Search string
	}{pageData, users, search}
	tmpl.Execute(w, data)
}

func (h *Handler) UserProfile(w http.ResponseWriter, req *http.Request) {
	if !h.router.IsAuthenticated(req) {
		http.Redirect(w, req, "/", http.StatusSeeOther)
		return
	}

	pageData := h.router.GetPageData(req)

	userID := strings.TrimPrefix(req.URL.Path, "/user/")

	var user models.User
	err := h.router.GetDB().QueryRow("SELECT id, username, role FROM users WHERE id = ?", userID).
		Scan(&user.ID, &user.Username, &user.Role)

	if err != nil {
		http.Error(w, "User not found", http.StatusNotFound)
		return
	}

	rows, _ := h.router.GetDB().Query("SELECT name, mac, ip FROM devices WHERE user_id = ?", userID)
	defer rows.Close()

	var devices []models.Device
	for rows.Next() {
		var d models.Device
		rows.Scan(&d.Name, &d.MAC, &d.IP)
		devices = append(devices, d)
	}

	tmpl := htmlTemplate.Must(htmlTemplate.ParseFiles("templates/user_profile.html"))
	data := struct {
		models.PageData
		ProfileUser models.User
		Devices     []models.Device
	}{pageData, user, devices}
	tmpl.Execute(w, data)
}

func (h *Handler) Ping(w http.ResponseWriter, req *http.Request) {
	if !h.router.IsAuthenticated(req) {
		http.Redirect(w, req, "/", http.StatusSeeOther)
		return
	}

	currentUser := h.router.GetCurrentUser(req)
	if currentUser.Role != "admin" {
		http.Error(w, "Access denied - admin required", http.StatusForbidden)
		return
	}

	pageData := h.router.GetPageData(req)

	var result string
	if req.Method == "POST" {
		host := req.FormValue("host")

		cmd := exec.Command("sh", "-c", "ping -c 4 "+host)
		output, err := cmd.CombinedOutput()
		if err != nil {
			result = "Error: " + err.Error()
		} else {
			result = string(output)
		}
	}

	tmpl := htmlTemplate.Must(htmlTemplate.ParseFiles("templates/ping.html"))
	data := struct {
		models.PageData
		Result string
	}{pageData, result}
	tmpl.Execute(w, data)
}

func (h *Handler) Settings(w http.ResponseWriter, req *http.Request) {
	if !h.router.IsAuthenticated(req) {
		http.Redirect(w, req, "/", http.StatusSeeOther)
		return
	}

	currentUser := h.router.GetCurrentUser(req)
	if currentUser.Role != "admin" {
		http.Error(w, "Access denied - admin required", http.StatusForbidden)
		return
	}

	pageData := h.router.GetPageData(req)
	var message string

	routerName := req.FormValue("router_name")
	if routerName != "" {
		_, err := h.router.GetDB().Exec("INSERT OR REPLACE INTO settings (key, value) VALUES ('router_name', ?)", routerName)
		if err != nil {
			message = "Error updating router name: " + err.Error()
		} else {
			message = "Router name updated to: " + routerName
		}
	}

	newPassword := req.FormValue("new_password")
	targetUser := req.FormValue("target_user")
	if newPassword != "" && targetUser != "" {
		_, err := h.router.GetDB().Exec("UPDATE users SET password = ? WHERE id = ?", newPassword, targetUser)
		if err != nil {
			message = "Error updating password: " + err.Error()
		} else {
			message = "Password updated successfully for user ID " + targetUser
		}
	}

	tmpl := textTemplate.Must(textTemplate.ParseFiles("templates/settings.html"))
	data := struct {
		models.PageData
		Message string
	}{pageData, message}
	tmpl.Execute(w, data)
}

func (h *Handler) Download(w http.ResponseWriter, req *http.Request) {
	w.Header().Set("Content-Disposition", "attachment; filename=interview-bundle.zip")
	w.Header().Set("Content-Type", "application/zip")
	http.ServeFile(w, req, "./interview-bundle.zip")
}

func (h *Handler) Logout(w http.ResponseWriter, req *http.Request) {
	http.SetCookie(w, &http.Cookie{
		Name:   "session",
		Value:  "",
		Path:   "/",
		MaxAge: -1,
	})
	http.Redirect(w, req, "/", http.StatusSeeOther)
}
